<?php
/**
 * Custom Order Status Management Page
 */

defined('ABSPATH') || exit;

// FontAwesome icons are handled via CSS classes and Unicode characters for security

// Fallback for sanitize_hex_color if not available (older WP versions)
if (!function_exists('sanitize_hex_color')) {
    function sanitize_hex_color($color) {
        if ('' === $color) {
            return '';
        }
        
        // 3 or 6 hex digits, or the empty string.
        if (preg_match('|^#([A-Fa-f0-9]{3}){1,2}$|', $color)) {
            return $color;
        }
        
        return '';
    }
}

// Note: $custom_statuses variable should be set by the admin class
// If not set, initialize as empty array
if (!isset($custom_statuses)) {
    $custom_statuses = array();
}
?>

<!-- CSS and JS are loaded via WordPress hooks -->

<div class="wrap kdsamref-custom-status-page">
    <!-- Modern Hero Header -->
    <div class="kdsamref-hero-header">
        <div class="kdsamref-hero-content">
            <div class="kdsamref-hero-left">
                <div class="kdsamref-hero-text">
                    <h1><?php esc_html_e('Order Status Management', 'kds-report-for-orderable'); ?></h1>
                    <p><?php esc_html_e('Create and manage custom order statuses for your KDS dashboard', 'kds-report-for-orderable'); ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Main Content Area -->
    <div class="kdsamref-main-content">
        <!-- Left Panel - Add New Status -->
        <div class="kdsamref-left-panel">
            <div class="kdsamref-panel-card">
                <div class="kdsamref-panel-header">
                    <div class="kdsamref-panel-icon">➕</div>
                    <h3><?php esc_html_e('Add New Status', 'kds-report-for-orderable'); ?></h3>
                </div>
                

                
                <form method="post" action="" class="kdsamref-status-form">
                    <?php wp_nonce_field('kdsamref_custom_status', '_wpnonce'); ?>
                    <input type="hidden" name="edit_slug" id="edit_slug" value="" />
                    
                    <div class="kdsamref-form-fields">
                        <!-- Basic Info Section -->
                        <div class="kdsamref-form-section">
                            <h4 class="kdsamref-section-title"><?php esc_html_e('Basic Information', 'kds-report-for-orderable'); ?></h4>
                            
                            <div class="kdsamref-field-row">
                                <div class="kdsamref-field-group">
                                    <label for="status_type"><?php esc_html_e('Type', 'kds-report-for-orderable'); ?></label>
                                    <select name="status_type" id="status_type" required>
                                        <option value="custom"><?php esc_html_e('Custom Status', 'kds-report-for-orderable'); ?></option>
                                        <?php
                                        $wc_statuses = wc_get_order_statuses();
                                        foreach ($wc_statuses as $status_key => $status_name) {
                                            $status_slug = str_replace('wc-', '', $status_key);
                                            echo '<option value="' . esc_attr($status_slug) . '">' . esc_html($status_name) . ' (Core)</option>';
                                        }
                                        ?>
                                    </select>
                                </div>
                                
                                <div class="kdsamref-field-group">
                                    <label for="status_slug"><?php esc_html_e('Slug', 'kds-report-for-orderable'); ?></label>
                                    <input type="text" id="status_slug" name="status_slug" 
                                           placeholder="ex: en-preparation" maxlength="20" required />
                                </div>
                            </div>
                            
                            <div class="kdsamref-field-group">
                                <label for="status_label"><?php esc_html_e('Display Name', 'kds-report-for-orderable'); ?></label>
                                <input type="text" id="status_label" name="status_label" 
                                       placeholder="ex: En Préparation" maxlength="50" required />
                            </div>
                        </div>
                        
                        <!-- Visual Customization Section -->
                        <div class="kdsamref-form-section">
                            <h4 class="kdsamref-section-title"><?php esc_html_e('Visual Customization', 'kds-report-for-orderable'); ?></h4>
                            
                            <div class="kdsamref-field-row">
                                <div class="kdsamref-field-group">
                                    <label for="status_color"><?php esc_html_e('Color', 'kds-report-for-orderable'); ?></label>
                                    <div class="kdsamref-color-field">
                                        <input type="color" id="status_color" name="status_color" value="#007cba" />
                                        <span class="kdsamref-color-label"><?php esc_html_e('Choose color', 'kds-report-for-orderable'); ?></span>
                                    </div>
                                </div>
                                
                                <div class="kdsamref-field-group">
                                    <label for="status_icon"><?php esc_html_e('Icon', 'kds-report-for-orderable'); ?></label>
                                    <div class="kdsamref-icon-field">
                                        <div class="kdsamref-icon-preview" id="icon_preview" role="button" tabindex="0">
                                            <i class="fas fa-cog"></i>
                                        </div>
                                        <span class="kdsamref-icon-label"><?php esc_html_e('Click to choose', 'kds-report-for-orderable'); ?></span>
                                    </div>
                                    <input type="hidden" id="status_icon" name="status_icon" value="fa-cog" />
                                    <input type="hidden" id="status_icon_family" name="status_icon_family" value="fontawesome" />
                                </div>
                            </div>
                        </div>
                        
                        <!-- Settings Section -->
                        <div class="kdsamref-form-section">
                            <h4 class="kdsamref-section-title"><?php esc_html_e('Settings', 'kds-report-for-orderable'); ?></h4>
                            
                            <div class="kdsamref-field-group">
                                <label class="kdsamref-toggle-label">
                                    <input type="checkbox" id="status_enabled" name="status_enabled" value="1" checked class="kdsamref-status-enabled-checkbox" />
                                    <span class="kdsamref-toggle-slider"></span>
                                    <?php esc_html_e('Enable this status', 'kds-report-for-orderable'); ?>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Form Actions -->
                        <div class="kdsamref-form-actions">
                            <button type="button" class="kdsamref-cancel-btn kdsamref-cancel-btn-hidden" id="cancel-btn">
                                <?php esc_html_e('Cancel', 'kds-report-for-orderable'); ?>
                            </button>
                            <button type="submit" name="submit" class="kdsamref-submit-btn">
                                <span class="kdsamref-btn-icon">💾</span>
                                <?php esc_html_e('Save Status', 'kds-report-for-orderable'); ?>
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Right Panel - Current Statuses -->
        <div class="kdsamref-right-panel">
            <div class="kdsamref-panel-card">
                <div class="kdsamref-panel-header">
                    <div class="kdsamref-panel-icon">📋</div>
                    <h3><?php esc_html_e('Current Statuses', 'kds-report-for-orderable'); ?></h3>
                    <div class="kdsamref-status-count">
                        <span class="kdsamref-count-number"><?php echo count($custom_statuses); ?></span>
                        <span class="kdsamref-count-label"><?php echo count($custom_statuses) === 1 ? 'status' : 'statuses'; ?></span>
                    </div>
                </div>
            
                <?php if (!empty($custom_statuses)): ?>
                    <div class="kdsamref-status-grid">
                        <?php foreach ($custom_statuses as $slug => $status_data): ?>
                            <div class="kdsamref-status-card" data-slug="<?php echo esc_attr($slug); ?>">
                                <div class="kdsamref-status-card-header">
                                    <div class="kdsamref-status-visual">
                                        <div class="kdsamref-status-color-dot" style="--status-color: <?php echo esc_attr($status_data['color']); ?>"></div>
                                        <?php if (!empty($status_data['icon'])): ?>
                                            <div class="kdsamref-status-icon-display">
                                                <?php 
                                                $icon_class = $status_data['icon'];
                                                $icon_family = $status_data['icon_family'] ?? 'fontawesome';
                                                $family_class = '';
                                                
                                                switch ($icon_family) {
                                                    case 'dashicons':
                                                        $family_class = 'dashicons';
                                                        break;
                                                    case 'woocommerce':
                                                        $family_class = 'wooicon';
                                                        break;
                                                    default:
                                                        $family_class = 'fa';
                                                        break;
                                                }
                                                ?>
                                                <i class="<?php echo esc_attr($family_class . ' ' . $icon_class . ' kdsamref-status-icon'); ?>" style="--status-color: <?php echo esc_attr($status_data['color']); ?>"></i>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="kdsamref-status-badges">
                                        <?php 
                                        $status_type = $status_data['status_type'] ?? 'custom';
                                        $enabled = isset($status_data['enabled']) ? $status_data['enabled'] : 1;
                                        ?>
                                        <span class="kdsamref-badge kdsamref-type-badge <?php echo esc_attr($status_type === 'custom' ? 'custom' : 'core'); ?>">
                                            <?php echo esc_html($status_type === 'custom' ? 'Custom' : 'Core'); ?>
                                        </span>
                                        <span class="kdsamref-badge kdsamref-status-badge <?php echo esc_attr($enabled ? 'active' : 'inactive'); ?>">
                                            <?php echo esc_html($enabled ? esc_html__('Active', 'kds-report-for-orderable') : esc_html__('Inactive', 'kds-report-for-orderable')); ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <div class="kdsamref-status-card-body">
                                    <h4 class="kdsamref-status-name"><?php echo esc_html($status_data['label']); ?></h4>
                                    <code class="kdsamref-status-slug"><?php echo esc_html($slug); ?></code>
                                </div>
                                
                                <div class="kdsamref-status-card-actions">
                                    <?php 
                                    $source = $status_data['source'] ?? 'custom';
                                    // Show edit/delete for custom statuses, show auto-synced label for Orderable Pro
                                    if ($source === 'orderable_pro'): ?>
                                        <div class="kdsamref-auto-sync-info">
                                            <span class="kdsamref-sync-icon">🔄</span>
                                            <?php esc_html_e('Auto-synced', 'kds-report-for-orderable'); ?>
                                        </div>
                                    <?php else: ?>
                                        <div class="kdsamref-action-buttons">
                                            <button class="kdsamref-action-btn kdsamref-toggle-btn <?php echo esc_attr($enabled ? 'kdsamref-active' : 'kdsamref-inactive'); ?>" 
                                                    data-slug="<?php echo esc_attr($slug); ?>" 
                                                    data-enabled="<?php echo esc_attr($enabled ? '1' : '0'); ?>"
                                                    title="<?php echo esc_attr($enabled ? esc_html__('Deactivate', 'kds-report-for-orderable') : esc_html__('Activate', 'kds-report-for-orderable')); ?>">
                                                <i class="fas <?php echo esc_attr($enabled ? 'fa-toggle-on' : 'fa-toggle-off'); ?>"></i>
                                            </button>
                                            
                                            <button class="kdsamref-action-btn kdsamref-edit-btn" data-slug="<?php echo esc_attr($slug); ?>" title="<?php esc_attr_e('Edit', 'kds-report-for-orderable'); ?>">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            
                                            <button class="kdsamref-action-btn kdsamref-delete-btn" data-slug="<?php echo esc_attr($slug); ?>" title="<?php esc_attr_e('Delete', 'kds-report-for-orderable'); ?>">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="kdsamref-empty-state">
                        <div class="kdsamref-empty-icon">📋</div>
                        <h4><?php esc_html_e('No Custom Statuses Yet', 'kds-report-for-orderable'); ?></h4>
                        <p><?php esc_html_e('Create your first custom status to get started with order management.', 'kds-report-for-orderable'); ?></p>
                        <div class="kdsamref-empty-actions">
                            <button type="button" class="kdsamref-empty-btn" onclick="document.getElementById('status_label').focus();">
                                <span class="kdsamref-btn-icon">➕</span>
                                <?php esc_html_e('Create First Status', 'kds-report-for-orderable'); ?>
                            </button>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modern Icon Picker Modal -->
    <div id="kdsamref-icon-modal" class="kdsamref-icon-modal">
        <div class="kdsamref-icon-modal-content">
            <div class="kdsamref-icon-modal-header">
                <h3><?php esc_html_e('Choose Icon', 'kds-report-for-orderable'); ?></h3>
                <button type="button" class="kdsamref-icon-modal-close">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="kdsamref-icon-modal-body">
                <div class="kdsamref-icon-search">
                    <input type="text" id="kdsamref-icon-search" placeholder="<?php esc_attr_e('Search icons...', 'kds-report-for-orderable'); ?>" class="kdsamref-icon-search-input">
                    <i class="fas fa-search kdsamref-icon-search-icon"></i>
                </div>
                <div class="kdsamref-icon-tabs">
                    <button type="button" class="kdsamref-icon-tab active" data-family="fontawesome">
                        <i class="fas fa-font"></i>
                        FontAwesome
                    </button>
                    <button type="button" class="kdsamref-icon-tab" data-family="dashicons">
                        <i class="dashicons dashicons-admin-generic"></i>
                        Dashicons
                    </button>
                </div>
                <div class="kdsamref-icon-grid" id="kdsamref-icon-grid">
                    <!-- Icons will be loaded here -->
                </div>
            </div>
        </div>
    </div>
</div>


