<?php
/**
 * Orderable Plugin Detection
 * Bu kodu includes/class-kdsamref-admin.php içinde kullanın
 */

class KDSAMREF_Plugin_Detector {
    
    /**
     * Check if Orderable plugin is active
     */
    public static function is_orderable_active() {
        // Method 1: Check if Orderable classes exist
        if (class_exists('Orderable') || class_exists('Orderable_Main')) {
            return true;
        }
        
        // Method 2: Check active plugins
        $active_plugins = get_option('active_plugins', array());
        foreach ($active_plugins as $plugin) {
            if (strpos($plugin, 'orderable') !== false) {
                return true;
            }
        }
        
        // Method 3: Check if Orderable functions exist
        if (function_exists('orderable_init') || function_exists('orderable')) {
            return true;
        }
        
        // Method 4: Check for Orderable tables in database
        global $wpdb;
        $orderable_tables = $wpdb->get_results(
            "SHOW TABLES LIKE '{$wpdb->prefix}orderable%'"
        );
        
        if (!empty($orderable_tables)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Get plugin compatibility mode
     */
    public static function get_compatibility_mode() {
        if (self::is_orderable_active()) {
            return 'orderable';
        }
        
        // Check if other restaurant plugins are active
        $restaurant_plugins = array(
            'woocommerce-food' => 'food',
            'restaurant-menu' => 'restaurant',
            'food-store' => 'food_store'
        );
        
        $active_plugins = get_option('active_plugins', array());
        foreach ($active_plugins as $plugin) {
            foreach ($restaurant_plugins as $plugin_slug => $mode) {
                if (strpos($plugin, $plugin_slug) !== false) {
                    return $mode;
                }
            }
        }
        
        return 'woocommerce'; // Default WooCommerce mode
    }
    
    /**
     * Get order compatibility data
     */
    public static function get_order_compatibility_info($order) {
        if (!($order instanceof WC_Order)) {
            return array();
        }
        
        $mode = self::get_compatibility_mode();
        
        switch ($mode) {
            case 'orderable':
                return self::get_orderable_order_data($order);
                
            case 'woocommerce':
            default:
                return self::get_woocommerce_order_data($order);
        }
    }
    
    /**
     * Get Orderable specific order data
     */
    private static function get_orderable_order_data($order) {
        // Orderable meta keys
        $orderable_meta = array(
            'orderable_service_type' => get_post_meta($order->get_id(), '_orderable_service_type', true),
            'orderable_location' => get_post_meta($order->get_id(), '_orderable_location', true),
            'orderable_scheduled_date' => get_post_meta($order->get_id(), '_orderable_scheduled_date', true),
            'orderable_table_number' => get_post_meta($order->get_id(), '_orderable_table_number', true)
        );
        
        return array(
            'plugin_mode' => 'orderable',
            'has_orderable_data' => !empty(array_filter($orderable_meta)),
            'orderable_meta' => $orderable_meta,
            'status_source' => 'orderable'
        );
    }
    
    /**
     * Get WooCommerce default order data
     */
    private static function get_woocommerce_order_data($order) {
        return array(
            'plugin_mode' => 'woocommerce',
            'has_orderable_data' => false,
            'orderable_meta' => array(),
            'status_source' => 'woocommerce',
            'shipping_method' => self::get_shipping_method_type($order)
        );
    }
    
    /**
     * Detect delivery type from WooCommerce shipping
     */
    private static function get_shipping_method_type($order) {
        $shipping_methods = $order->get_shipping_methods();
        
        if (empty($shipping_methods)) {
            return 'dine-in'; // No shipping = dine-in
        }
        
        foreach ($shipping_methods as $method) {
            $method_id = $method->get_method_id();
            
            if (strpos($method_id, 'pickup') !== false || strpos($method_id, 'collection') !== false) {
                return 'pickup';
            } elseif (strpos($method_id, 'delivery') !== false) {
                return 'delivery';
            }
        }
        
        return 'delivery'; // Default
    }
}

// Usage example:
// $compatibility = KDSAMREF_Plugin_Detector::get_compatibility_mode();
// $order_info = KDSAMREF_Plugin_Detector::get_order_compatibility_info($order);
?>
