<?php
/**
 * AJAX Handler Class
 * 
 * Handles AJAX requests for the restaurant orders dashboard
 */

if (!defined('ABSPATH')) {
    exit;
}

class KDSAMREF_Ajax_Handler {

    /**
     * Constructor
     */
    public function __construct() {
        // Add AJAX hooks
        // Centralized AJAX registrations
        add_action('wp_ajax_kdsamref_get_orders', array($this, 'get_orders'));
        add_action('wp_ajax_kdsamref_update_order_status', array($this, 'update_order_status'));
        add_action('wp_ajax_kdsamref_get_order_stats', array($this, 'get_order_stats'));
        add_action('wp_ajax_kdsamref_add_order_note', array($this, 'add_order_note'));
        add_action('wp_ajax_kdsamref_get_single_order', array($this, 'get_single_order'));
        add_action('wp_ajax_kdsamref_search_orders', array($this, 'search_orders'));
        add_action('wp_ajax_kdsamref_get_orders_by_date_range', array($this, 'get_orders_by_date_range'));

        add_action('wp_ajax_kdsamref_get_settings', array($this, 'get_settings'));
        
        // Custom status management AJAX hooks
        add_action('wp_ajax_kdsamref_save_custom_status', array($this, 'save_custom_status'));
        add_action('wp_ajax_kdsamref_delete_custom_status', array($this, 'delete_custom_status'));
        add_action('wp_ajax_kdsamref_get_custom_statuses', array($this, 'get_custom_statuses'));
        add_action('wp_ajax_kdsamref_toggle_custom_status', array($this, 'toggle_custom_status'));
        add_action('wp_ajax_kdsamref_sync_orderable_pro_statuses', array($this, 'sync_orderable_pro_statuses'));

        // Stock management AJAX hooks
        add_action('wp_ajax_kdsamref_get_stock_items', array($this, 'get_stock_items'));
        add_action('wp_ajax_kdsamref_update_stock_quantity', array($this, 'update_stock_quantity'));
    }

    /**
     * Enhanced security verification with rate limiting and logging
     */
    private function verify_request() {
        // Debug logging for AJAX requests
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $action = sanitize_text_field(wp_unslash($_POST['action'] ?? 'unknown'));
            $ip = $this->get_client_ip();
            error_log("WRO AJAX Request: {$action} from IP {$ip}");
        }
        
        // Rate limiting check
        $this->check_rate_limit();
        
        // Enhanced nonce verification with proper sanitization
        if (!isset($_POST['nonce'])) {
            $this->log_security_event('missing_nonce', [
                'ip' => $this->get_client_ip(),
                'user_agent' => sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'] ?? '')),
                'endpoint' => sanitize_text_field(wp_unslash($_POST['action'] ?? 'unknown'))
            ]);
            wp_send_json_error(array(
                'message' => esc_html__('Security verification required', 'kds-report-for-orderable')
            ));
        }
        
        $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        if (!wp_verify_nonce($nonce, 'kdsamref_nonce')) {
            $this->log_security_event('invalid_nonce', [
                'ip' => $this->get_client_ip(),
                'user_id' => get_current_user_id(),
                'nonce_provided' => sanitize_text_field(wp_unslash($_POST['nonce'])),
                'endpoint' => sanitize_text_field(wp_unslash($_POST['action'] ?? 'unknown'))
            ]);
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed', 'kds-report-for-orderable')
            ));
        }

        // Enhanced user permission check with capability hierarchy
        $required_caps = [
            'manage_kdsamref_orders',
            'manage_woocommerce', 
            'shop_manager',
            'gestion_commandes',
            'order_manager',
            'administrator'
        ];
        
        $user_can_access = false;
        foreach ($required_caps as $cap) {
            if (current_user_can($cap)) {
                $user_can_access = true;
                break;
            }
        }
        
        if (!$user_can_access) {
            $this->log_security_event('insufficient_permissions', [
                'ip' => $this->get_client_ip(),
                'user_id' => get_current_user_id(),
                'user_caps' => wp_get_current_user()->allcaps,
                'endpoint' => sanitize_text_field(wp_unslash($_POST['action'] ?? 'unknown'))
            ]);
            wp_send_json_error(array(
                'message' => esc_html__('Permission denied', 'kds-report-for-orderable')
            ));
        }
    }

    /**
     * Get stock items for stock management
     */
    public function get_stock_items() {
        $this->verify_request();

        try {
            $stock_items = array();

            // Check if WooCommerce is active
            if (!class_exists('WooCommerce')) {
                wp_send_json_error(array(
                    'message' => esc_html__('WooCommerce is not active', 'kds-report-for-orderable')
                ));
                return;
            }

            $args = array(
                'post_type' => 'product',
                'post_status' => 'publish',
                'posts_per_page' => -1,
                'meta_query' => array(
                    array(
                        'key' => '_manage_stock',
                        'value' => 'yes',
                        'compare' => '='
                    )
                )
            );

            $products = get_posts($args);

            foreach ($products as $product) {
                $wc_product = wc_get_product($product->ID);
                if (!$wc_product) continue;

                $stock_quantity = $wc_product->get_stock_quantity();
                $stock_status = $wc_product->get_stock_status();
                $price = $wc_product->get_price();
                $sku = $wc_product->get_sku();
                $categories = get_the_terms($product->ID, 'product_cat');
                $category_name = $categories ? $categories[0]->name : '';

                // Determine status based on quantity
                $status = 'in-stock';
                if ($stock_quantity === 0 || $stock_status === 'outofstock') {
                    $status = 'out-of-stock';
                } elseif ($stock_quantity <= 5) {
                    $status = 'low-stock';
                }

                $stock_items[] = array(
                    'id' => $product->ID,
                    'name' => $product->post_title,
                    'sku' => $sku ?: 'SKU-' . $product->ID,
                    'category' => $category_name,
                    'price' => wc_price($price),
                    'quantity' => $stock_quantity !== null ? $stock_quantity : 0,
                    'status' => $status
                );
            }

            wp_send_json_success(array(
                'stock_items' => $stock_items
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('Failed to load stock items', 'kds-report-for-orderable'),
                'error' => $e->getMessage()
            ));
        }
    }

    /**
     * Update stock quantity
     */
    public function update_stock_quantity() {
        $this->verify_request();

        try {
            $product_id = intval($_POST['product_id']);
            $new_quantity = intval($_POST['quantity']);

            if ($product_id <= 0) {
                wp_send_json_error(array(
                    'message' => esc_html__('Invalid product ID', 'kds-report-for-orderable')
                ));
            }

            if ($new_quantity < 0) {
                wp_send_json_error(array(
                    'message' => esc_html__('Quantity cannot be negative', 'kds-report-for-orderable')
                ));
            }

            // Check if WooCommerce is active
            if (!class_exists('WooCommerce')) {
                wp_send_json_error(array(
                    'message' => esc_html__('WooCommerce is not active', 'kds-report-for-orderable')
                ));
            }

            $product = wc_get_product($product_id);
            if (!$product) {
                wp_send_json_error(array(
                    'message' => esc_html__('Product not found', 'kds-report-for-orderable')
                ));
            }

            // Update stock quantity
            $product->set_stock_quantity($new_quantity);
            
            // Update stock status based on quantity
            if ($new_quantity === 0) {
                $product->set_stock_status('outofstock');
            } elseif ($new_quantity <= 5) {
                $product->set_stock_status('onbackorder');
            } else {
                $product->set_stock_status('instock');
            }

            $product->save();

            // Get updated product data
            $stock_quantity = $product->get_stock_quantity();
            $stock_status = $product->get_stock_status();
            
            // Determine status for response
            $status = 'in-stock';
            if ($stock_quantity === 0 || $stock_status === 'outofstock') {
                $status = 'out-of-stock';
            } elseif ($stock_quantity <= 5) {
                $status = 'low-stock';
            }

            wp_send_json_success(array(
                'message' => esc_html__('Stock quantity updated successfully', 'kds-report-for-orderable'),
                'product_id' => $product_id,
                'new_quantity' => $stock_quantity,
                'new_status' => $status
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('Failed to update stock quantity', 'kds-report-for-orderable'),
                'error' => $e->getMessage()
            ));
        }
    }

    // Placeholder methods for other AJAX functions
    public function get_orders() {
        // Implementation for getting orders
    }

    public function update_order_status() {
        // Implementation for updating order status
    }

    public function get_order_stats() {
        // Implementation for getting order stats
    }

    public function add_order_note() {
        // Implementation for adding order notes
    }

    public function get_single_order() {
        // Implementation for getting single order
    }

    public function search_orders() {
        // Implementation for searching orders
    }

    public function get_orders_by_date_range() {
        // Implementation for getting orders by date range
    }

    public function get_settings() {
        // Implementation for getting settings
    }

    public function save_custom_status() {
        // Implementation for saving custom status
    }

    public function delete_custom_status() {
        // Implementation for deleting custom status
    }

    public function get_custom_statuses() {
        // Implementation for getting custom statuses
    }

    public function toggle_custom_status() {
        // Implementation for toggling custom status
    }

    public function sync_orderable_pro_statuses() {
        // Implementation for syncing Orderable Pro statuses
    }

    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }

    /**
     * Check rate limiting
     */
    private function check_rate_limit() {
        // Basic rate limiting implementation
        $ip = $this->get_client_ip();
        $rate_limit_key = 'kdsamref_rate_limit_' . md5($ip);
        $rate_limit_data = get_transient($rate_limit_key);
        
        if ($rate_limit_data === false) {
            set_transient($rate_limit_key, array('count' => 1, 'first_request' => time()), 60);
        } else {
            $rate_limit_data['count']++;
            if ($rate_limit_data['count'] > 100) { // 100 requests per minute
                wp_send_json_error(array(
                    'message' => esc_html__('Rate limit exceeded', 'kds-report-for-orderable')
                ));
            }
            set_transient($rate_limit_key, $rate_limit_data, 60);
        }
    }

    /**
     * Log security events
     */
    private function log_security_event($event_type, $data) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("KDSAMREF Security Event: {$event_type} - " . json_encode($data));
        }
    }
}
